/* FILE: mlexer.c                               (D. Tottingham  12/26/90)

This is a collection of C functions that lexes the control file for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

lx_initialize ()                initialize the lexer
lx_match ()                     match a token with the lookahead
lx_set_stream ()                set the input stream

EXTERNAL FUNCTIONS CALLED:

o_write_logfile ()              write string to log file stream

HISTORY:
   none

*/



/*************************************************************************
                             INCLUDE FILES

*************************************************************************/
#include <stdio.h>
#include <ctype.h>

#include "mconst.h"
#include "mlexer.h"
#include "mlog.h"


/*************************************************************************
                              GLOBAL DATA

**************************************************************************/
PRIVATE LX_KEYWORD keywords[] = {
  {"AuthorityCode"            ,  C_AUTHORITY             },
  {"AutoLocation"             ,  C_AUTO_LOCATION         },
  {"AutoLog"                  ,  C_AUTO_LOG              },
  {"Autoreboot"               ,  C_AUTO_REBOOT           },
  {"Autotrigger"              ,  C_AUTO_TRIGGER          },
  {"BandLimits"               ,  C_BAND_LIMITS           },
  {"BandRecording"            ,  C_BAND_RECORDING        },
  {"BuffersToAverage"         ,  C_BUFFERS_TO_AVERAGE    },
  {"CalibrationRecording"     ,  C_CALIBRATION_RECORDING },
  {"Ch"                       ,  C_CH                    },
  {"ChannelBlocksize"         ,  C_CHANNEL_BLOCKSIZE     },
  {"ChannelGain"              ,  C_CHANNEL_GAIN          },
  {"ClockSource"              ,  C_CLOCK_SOURCE          },
  {"Component"                ,  C_COMPONENT             },
  {"CriticalAlpha"            ,  C_CRITICAL_ALPHA        },
  {"CriticalBeta"             ,  C_CRITICAL_BETA         },
  {"CriticalGamma"            ,  C_CRITICAL_GAMMA        },
  {"CriticalMu"               ,  C_CRITICAL_MU           },
  {"CriticalNu"               ,  C_CRITICAL_NU           },
  {"CriticalPhi"              ,  C_CRITICAL_PHI          },
  {"DigitizationRate"         ,  C_DIGITIZATION_RATE     },
  {"Display"                  ,  C_DISPLAY               },
  {"EventAlertBell"           ,  C_EVENT_ALERT_BELL      },
  {"EventContinuationCount"   ,  C_EVENT_CONTINUE_COUNT  },
  {"EXTERNAL"                 ,  C_EXTERNAL              },
  {"FALSE"                    ,  C_FALSE                 },
  {"FreerunBlockTime"         ,  C_FREERUN_BLOCK_TIME    },
  {"Gain"                     ,  C_GAIN                  },
  {"HalfspaceVelocity"        ,  C_HALFSPACE_VELOCITY    },
  {"INTERNAL"                 ,  C_INTERNAL              },
  {"LogPathName"              ,  C_LOG_PATHNAME          },
  {"LTAverageWindow"          ,  C_LTA_WINDOW            },
  {"MaxCalibrationTime"       ,  C_MAX_CALIBRATION_TIME  },
  {"MaxEventTime"             ,  C_MAX_EVENT_TIME        },
  {"MinEventTime"             ,  C_MIN_EVENT_TIME        },
  {"NetworkName"              ,  C_NETWORK               },
  {"NetworkNodeId"            ,  C_NETWORK_NODE_ID       },
  {"OFF"                      ,  C_OFF                   },
  {"ON"                       ,  C_ON                    },
  {"PathName"                 ,  C_PATHNAME              },
  {"PreEventTime"             ,  C_PRE_EVENT_TIME        },
  {"STAverageWindow"          ,  C_STA_WINDOW            },
  {"StName"                   ,  C_STNAME                },
  {"Time"                     ,  C_TIME                  },
  {"Trigger"                  ,  C_TRIGGER               },
  {"TriggerConfirmationCount" ,  C_TRIGGER_CONFIRM_COUNT },
  {"TriggerSource"            ,  C_TRIGGER_SOURCE        },
  {"TriggerTimeLimit"         ,  C_TRIGGER_TIME_LIMIT    },
  {"TRUE"                     ,  C_TRUE                  },
  { NULL                      ,  -1                      }
};

PRIVATE FILE * stream;
PRIVATE char out_str[85];

LX_INFO lex_info;


/*=======================================================================*
 *                               lexer                                   *
 *=======================================================================*/
/* Get a token from the input stream.                                    */

PRIVATE
void lexer()
{
   int ch, index;

   /* Munch white space and comments */
   while (isspace(lex_info.lexeme[0] = ch = fgetc(stream)) ||
          ch == COMMENT_CHAR) {
      if (ch == '\n')
         lex_info.lineno++;
      else if (ch == COMMENT_CHAR) {
         while ((lex_info.lexeme[0] = ch = fgetc(stream)) != '\n');
         lex_info.lineno++;
      }
   }
   lex_info.lexeme[1] = '\0';

   if (ch == EOF)
      lex_info.lookahead = C_DONE;

   else if (isdigit(lex_info.lexeme[0])) {
      lex_info.lookahead = C_CONSTANT;
      fscanf(stream, "%[0123456789.]", lex_info.lexeme+1);

   } else if (isalpha(lex_info.lexeme[0])) {
      lex_info.lookahead = C_ID;
      for (index = 1; index < MAXLEX-1 && isalnum(ch = fgetc(stream)); index++)
         lex_info.lexeme[index] = ch;

      lex_info.lexeme[index] = '\0';
      if (index == MAXLEX-1) {
         sprintf (out_str, "WARNING: Command %s has been truncated to ",
                           "maximum command length.\n", lex_info.lexeme);
         o_write_logfile (out_str);
         printf (out_str);
      }
      ungetc (ch, stream);
      for (index = 0; keywords[index].token != -1; index++)
         if (!strcmp(keywords[index].name, lex_info.lexeme))
            lex_info.lookahead = keywords[index].token;

   } else if (lex_info.lexeme[0] == '"') {
      lex_info.lookahead = C_STRING;
      for (index = 0; index < MAXLEX-1 && (ch = fgetc(stream)) != '"'; index++)
         lex_info.lexeme[index] = ch;
      lex_info.lexeme[index] = '\0';

   } else if (lex_info.lexeme[0] == '\'') {
      lex_info.lookahead = C_CHARACTER;
      lex_info.lexeme[0] = fgetc(stream);
      lex_info.lexeme[1] = '\0';
      if ((ch = fgetc (stream)) != '\'') {
         sprintf (out_str, "WARNING: Invalid character constant on line %d\n", lex_info.lineno);
         o_write_logfile (out_str);
         printf (out_str);
      }
   } else lex_info.lookahead = *lex_info.lexeme;
}

/*=======================================================================*
 *                           lx_initialize                               *
 *=======================================================================*/
/* Initialize the lexer.                                                 */

PUBLIC
void lx_initialize ()
{
   lex_info.lookahead = 0;
   lex_info.lineno = 1;
   lex_info.error = 0;
}

/*=======================================================================*
 *                             lx_match                                  *
 *=======================================================================*/
/* Match a token with the lookahead.                                     */

PUBLIC
void lx_match (token)
int token;
{
   if (lex_info.error) {
      if (token == ';') {
         lex_info.error = OFF;
         lexer();
      }

   } else if (lex_info.lookahead != token) {
      sprintf(out_str, "WARNING: Syntax error on line %d of control file.\n", lex_info.lineno);
      o_write_logfile (out_str);
      printf (out_str);
      while (lex_info.lookahead != ';' && lex_info.lookahead != C_DONE)
         lexer();
      lex_info.error = ON;

   } else lexer();
}

/*=======================================================================*
 *                           lx_set_stream                               *
 *=======================================================================*/
/* Set the input stream.                                                 */

PUBLIC
void lx_set_stream (the_stream)
FILE * the_stream;
{
   stream = the_stream;
}
